/*____________________________________________________________________________
		Copyright (C) 2000 Network Associates, Inc.
        All rights reserved.

        $Id: CIrp.h,v 1.3 1999/12/18 06:21:48 nryan Exp $
____________________________________________________________________________*/

#ifndef Included_CIrp_h	// [
#define Included_CIrp_h

#include "pgpClassesConfig.h"

_PGP_BEGIN

// Class CIrp

class CIrp
{
	NOT_COPYABLE(CIrp)

private:
	enum IrpStackLocation {Current, Next};

public:
	CIrp();
	CIrp(PIRP pIrp);

	operator IRP&() {return *mIrp;}
	operator const IRP&() const {return *mIrp;}

	operator PIRP() {return mIrp;}
	operator const PIRP() const {return mIrp;}

	PIRP		operator->() {return mIrp;}
	const PIRP	operator->() const {return mIrp;}

	PIRP		Get() const
	{
		return mIrp;
	}

	NTSTATUS& 	Status() const 
	{
		return mIrp->IoStatus.Status;
	}

	ULONG&		Information() const 
	{
		return mIrp->IoStatus.Information;
	}

	PMDL&		MdlAddress() const 
	{
		return mIrp->MdlAddress;
	}

	PVOID&		UserBuffer() const 
	{
		return mIrp->UserBuffer;
	}

	PVOID&		DriverContext1() const 
	{
		return mIrp->Tail.Overlay.DriverContext[1];
	}

	PVOID&		DriverContext2() const 
	{
		return mIrp->Tail.Overlay.DriverContext[2];
	}

	PVOID&		DriverContext3() const 
	{
		return mIrp->Tail.Overlay.DriverContext[3];
	}

	PVOID&		DriverContext4() const 
	{
		return mIrp->Tail.Overlay.DriverContext[4];
	}

	PGPUInt8&	MajorFunction() const 
	{
		return StackLocation()->MajorFunction;
	}

	PGPUInt8&	MinorFunction() const 
	{
		return StackLocation()->MinorFunction;
	}

	PGPUInt8&	Flags() const 
	{
		return StackLocation()->Flags;
	}

	PGPUInt8&	Control() const 
	{
		return StackLocation()->Control;
	}

	ULONG&		ReadLength() const 
	{
		return StackLocation()->Parameters.Read.Length;
	}

	PGPInt64&	ReadPos() const 
	{
		return StackLocation()->Parameters.Read.ByteOffset.QuadPart;
	}

	ULONG&		WriteLength() const 
	{
		return StackLocation()->Parameters.Write.Length;
	}

	PGPInt64&	WritePos() const 
	{
		return StackLocation()->Parameters.Write.ByteOffset.QuadPart;
	}

	ULONG&		IoctlCode() const 
	{
		return StackLocation()->Parameters.DeviceIoControl.IoControlCode;
	}

	void *&		IoctlOutputBuffer() const 
	{
		return mIrp->UserBuffer;
	}

	ULONG&		IoctlOutputBufferLength() const 
	{
		return StackLocation()->Parameters.DeviceIoControl.
			OutputBufferLength;
	}

	void *&		IoctlInputBuffer() const 
	{
		return mIrp->AssociatedIrp.SystemBuffer;
	}

	ULONG&		IoctlInputBufferLength() const 
	{
		return StackLocation()->Parameters.DeviceIoControl.InputBufferLength;
	}

	void *&		IoctlType3InputBuffer() const 
	{
		return StackLocation()->Parameters.DeviceIoControl.Type3InputBuffer;
	}

	PDEVICE_OBJECT&	DeviceObject() const  
	{
		return StackLocation()->DeviceObject;
	}

	PFILE_OBJECT&	FileObject() const  
	{
		return StackLocation()->FileObject;
	}

	PIO_STACK_LOCATION	StackLocation() const;

	void	UseCurrentStackLocation() 
	{
		mStackLocation = Current;
	}

	void	UseNextStackLocation() 
	{
		mStackLocation = Next;
	}

#if (_WIN32_WINNT >= 0x5000)
	void	StartNextPowerIrp() const
	{
		PoStartNextPowerIrp(mIrp);
	}
#endif	// _WIN32_WINNT >= 0x5000

	NTSTATUS	Complete(NTSTATUS status, CCHAR boost = IO_NO_INCREMENT);
	void		SetCancelRoutine(PDRIVER_CANCEL cancelFunc);
	void		MarkPending();
	void		UnmarkPending();

	CComboError	BuildDeviceIoControlRequest(PGPUInt32 ioctlCode, 
		PDEVICE_OBJECT pDeviceObject, const void *inBuf, 
		PGPUInt32 sizeInBuf, void *outBuf, PGPUInt32 sizeOutBuf, 
		PGPBoolean internal, KEVENT& event, IO_STATUS_BLOCK& ioStatus);

	CComboError	BuildSynchronousFsdRequest(PGPUInt32 ioctlCode, 
		PDEVICE_OBJECT pDeviceObject, const void *buf, PGPUInt32 nBytes, 
		PGPUInt64 pos, KEVENT& event, IO_STATUS_BLOCK& ioStatus);

private:
	PIRP				mIrp;
	IrpStackLocation	mStackLocation;
};


// Class CIrp member functions

inline 
CIrp::CIrp() : mIrp(NULL)
{
	UseCurrentStackLocation();
}

inline 
CIrp::CIrp(PIRP pIrp) : mIrp(pIrp)
{
	pgpAssertAddrValid(pIrp, IRP);
	UseCurrentStackLocation();
}

inline 
PIO_STACK_LOCATION 
CIrp::StackLocation() const
{
	if (mStackLocation == Current)
		return IoGetCurrentIrpStackLocation(mIrp);
	else
		return IoGetNextIrpStackLocation(mIrp);
}

inline 
NTSTATUS 
CIrp::Complete(NTSTATUS status, CCHAR boost)
{
	Status() = status;
	IoCompleteRequest(mIrp, boost);

	return status;
}

inline 
void 
CIrp::SetCancelRoutine(PDRIVER_CANCEL cancelFunc)
{
	IoSetCancelRoutine(mIrp, cancelFunc);
}

inline 
void 
CIrp::MarkPending()
{
	IoMarkIrpPending(mIrp);
}

inline 
void 
CIrp::UnmarkPending()
{
    IoGetCurrentIrpStackLocation(mIrp)->Control &= ~SL_PENDING_RETURNED;
}

inline 
CComboError 
CIrp::BuildDeviceIoControlRequest(
	PGPUInt32			ioctlCode, 
	PDEVICE_OBJECT		pDeviceObject, 
	const void			*inBuf, 
	PGPUInt32			sizeInBuf, 
	void				*outBuf, 
	PGPUInt32			sizeOutBuf, 
	PGPBoolean			internal, 
	KEVENT&				event, 
	IO_STATUS_BLOCK&	ioStatus)
{
	pgpAssertAddrValid(pDeviceObject, DEVICE_OBJECT);

	CComboError	error;

	mIrp = IoBuildDeviceIoControlRequest(ioctlCode, pDeviceObject, 
		const_cast<void *>(inBuf), sizeInBuf, outBuf, sizeOutBuf, internal, 
		&event, &ioStatus);

	if (IsNull(mIrp))
		error.pgpErr = kPGPError_NTDrvObjectOpFailed;

	return error;
}

inline 
CComboError 
CIrp::BuildSynchronousFsdRequest(
		PGPUInt32			majorFunc, 
		PDEVICE_OBJECT		pDeviceObject, 
		const void			*buf, 
		PGPUInt32			nBytes, 
		PGPUInt64			pos, 
		KEVENT&				event, 
		IO_STATUS_BLOCK&	ioStatus)
{
	pgpAssertAddrValid(pDeviceObject, DEVICE_OBJECT);

	CComboError	error;
	LARGE_INTEGER	liPos;

	liPos.QuadPart = pos;

	mIrp = IoBuildSynchronousFsdRequest(majorFunc, pDeviceObject, 
		const_cast<void *>(buf), nBytes, &liPos, &event, &ioStatus);

	if (IsNull(mIrp))
		error.pgpErr = kPGPError_NTDrvObjectOpFailed;

	return error;
}

_PGP_END

#endif	// ] Included_CIrp_h
